using System;
using System.Windows.Forms;

using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.DAL.VistALink.OpenLibrary;
using gov.va.med.vbecs.GUI.controls;

namespace gov.va.med.vbecs.GUI
{
	/// <summary>
	/// Base dialog for all VistA (UC18) related dialogs.
	/// </summary>
	public class DlgVistABase : VbecsBaseForm
	{
		private StatusBar sbFormStatusBar;
		private StatusBarPanel spnlServerName;
		private StatusBarPanel spnlVolumeName;
		private StatusBarPanel spnlUCI;
		private StatusBarPanel spnlDeviceName;
		private StatusBarPanel spnPortNumber;
		private System.ComponentModel.IContainer components = null;

		private bool _firstTimeActivationFlag;
		private RemoteServerSetupAndIntroInfo _serverInfo;
		private ServerConnectionInfo _serverConnectionInfo;
		private Control _initFocusPoint;
		private bool _errorProviderDirty;

		/// <summary>
		/// Default constructor added for VS.NET designer support.
		/// </summary>
		internal DlgVistABase() : base( false )
		{
			InitializeComponent();
		}

		/// <summary>
		/// Actual constructor that should be used by the application.
		/// </summary>
		/// <param name="serverInfo">Remote VistA server info.</param>
		/// <param name="serverConnectionInfo"></param>
		public DlgVistABase( RemoteServerSetupAndIntroInfo serverInfo, ServerConnectionInfo serverConnectionInfo ) : this()
		{
			if( serverInfo == null )
				throw( new ArgumentNullException( "serverInfo" ) );

			if( serverConnectionInfo == null )
				throw( new ArgumentNullException( "serverConnectionInfo" ) );

			_serverInfo = serverInfo;
			_serverConnectionInfo = serverConnectionInfo;

			this.DialogResult = DialogResult.None;
			_firstTimeActivationFlag = true;
			_errorProviderDirty = false;
		}

		/// <summary>
		/// Late binding for shared OK and Cancel buttons and 
		/// control to set focus on initially.
		/// </summary>
		/// <param name="okButton">OK button.</param>
		/// <param name="cancelButton">Cancel button.</param>
		/// <param name="initialFocusPoint">Control to set focus on initially.</param>
		protected void SetControlBindings( VbecsButton okButton, VbecsButton cancelButton, Control initialFocusPoint )
		{
			if( okButton != null )
			{
				this.AcceptButton = okButton;
				okButton.Click += new System.EventHandler( this.btnOk_Click );
			}

			if( cancelButton != null )
			{
				this.CancelButton = cancelButton;
				cancelButton.Click += new System.EventHandler( this.btnCancel_Click );
			}

			if( initialFocusPoint != null )
				this._initFocusPoint = initialFocusPoint;
		}

		/// <summary>
		/// Clean up any resources being used.
		/// </summary>
		protected override void Dispose( bool disposing )
		{
			if( disposing )
			{
				if (components != null) 
				{
					components.Dispose();
				}
			}
			base.Dispose( disposing );
		}

		#region Designer generated code
		/// <summary>
		/// Required method for Designer support - do not modify
		/// the contents of this method with the code editor.
		/// </summary>
		private void InitializeComponent()
		{
            System.ComponentModel.ComponentResourceManager resources = new System.ComponentModel.ComponentResourceManager(typeof(DlgVistABase));
            this.sbFormStatusBar = new System.Windows.Forms.StatusBar();
            this.spnlServerName = new System.Windows.Forms.StatusBarPanel();
            this.spnlVolumeName = new System.Windows.Forms.StatusBarPanel();
            this.spnlUCI = new System.Windows.Forms.StatusBarPanel();
            this.spnlDeviceName = new System.Windows.Forms.StatusBarPanel();
            this.spnPortNumber = new System.Windows.Forms.StatusBarPanel();
            ((System.ComponentModel.ISupportInitialize)(this.errError)).BeginInit();
            ((System.ComponentModel.ISupportInitialize)(this.spnlServerName)).BeginInit();
            ((System.ComponentModel.ISupportInitialize)(this.spnlVolumeName)).BeginInit();
            ((System.ComponentModel.ISupportInitialize)(this.spnlUCI)).BeginInit();
            ((System.ComponentModel.ISupportInitialize)(this.spnlDeviceName)).BeginInit();
            ((System.ComponentModel.ISupportInitialize)(this.spnPortNumber)).BeginInit();
            this.SuspendLayout();
            // 
            // vbecsLogo1
            // 
            this.vbecsLogo1.Size = new System.Drawing.Size(242, 30);
            // 
            // lblRequiredField
            // 
            this.lblRequiredField.Location = new System.Drawing.Point(0, 85);
            // 
            // sbFormStatusBar
            // 
            this.sbFormStatusBar.Location = new System.Drawing.Point(0, 99);
            this.sbFormStatusBar.Name = "sbFormStatusBar";
            this.sbFormStatusBar.Panels.AddRange(new System.Windows.Forms.StatusBarPanel[] {
            this.spnlServerName,
            this.spnlVolumeName,
            this.spnlUCI,
            this.spnlDeviceName,
            this.spnPortNumber});
            this.sbFormStatusBar.ShowPanels = true;
            this.sbFormStatusBar.Size = new System.Drawing.Size(242, 24);
            this.sbFormStatusBar.SizingGrip = false;
            this.sbFormStatusBar.TabIndex = 1;
            // 
            // spnlServerName
            // 
            this.spnlServerName.AutoSize = System.Windows.Forms.StatusBarPanelAutoSize.Spring;
            this.spnlServerName.Name = "spnlServerName";
            this.spnlServerName.Text = "Server:";
            this.spnlServerName.ToolTipText = "VistA server name";
            this.spnlServerName.Width = 59;
            // 
            // spnlVolumeName
            // 
            this.spnlVolumeName.AutoSize = System.Windows.Forms.StatusBarPanelAutoSize.Contents;
            this.spnlVolumeName.Name = "spnlVolumeName";
            this.spnlVolumeName.Text = "Volume:";
            this.spnlVolumeName.ToolTipText = "VistA volume name";
            this.spnlVolumeName.Width = 56;
            // 
            // spnlUCI
            // 
            this.spnlUCI.AutoSize = System.Windows.Forms.StatusBarPanelAutoSize.Contents;
            this.spnlUCI.Name = "spnlUCI";
            this.spnlUCI.Text = "UCI:";
            this.spnlUCI.ToolTipText = "VistA UCI";
            this.spnlUCI.Width = 37;
            // 
            // spnlDeviceName
            // 
            this.spnlDeviceName.AutoSize = System.Windows.Forms.StatusBarPanelAutoSize.Contents;
            this.spnlDeviceName.Name = "spnlDeviceName";
            this.spnlDeviceName.Text = "Device:";
            this.spnlDeviceName.ToolTipText = "VistA device name";
            this.spnlDeviceName.Width = 52;
            // 
            // spnPortNumber
            // 
            this.spnPortNumber.AutoSize = System.Windows.Forms.StatusBarPanelAutoSize.Contents;
            this.spnPortNumber.Name = "spnPortNumber";
            this.spnPortNumber.Text = "Port:";
            this.spnPortNumber.ToolTipText = "VistA server port number";
            this.spnPortNumber.Width = 38;
            // 
            // DlgVistABase
            // 
            this.AutoScaleBaseSize = new System.Drawing.Size(5, 13);
            this.ClientSize = new System.Drawing.Size(242, 123);
            this.Controls.Add(this.sbFormStatusBar);
            this.hlphelpChm.SetHelpNavigator(this, System.Windows.Forms.HelpNavigator.Topic);
            this.Icon = ((System.Drawing.Icon)(resources.GetObject("$this.Icon")));
            this.MinimizeBox = false;
            this.MinimumSize = new System.Drawing.Size(250, 150);
            this.Name = "DlgVistABase";
            this.RequiredFieldsLabel = true;
            this.hlphelpChm.SetShowHelp(this, true);
            this.TopMost = true;
            this.Activated += new System.EventHandler(this.VistASignOnDialog_Activated);
            this.KeyDown += new System.Windows.Forms.KeyEventHandler(this.DlgVistABase_KeyDown);
            this.Controls.SetChildIndex(this.sbFormStatusBar, 0);
            this.Controls.SetChildIndex(this.vbecsLogo1, 0);
            this.Controls.SetChildIndex(this.lblRequiredField, 0);
            ((System.ComponentModel.ISupportInitialize)(this.errError)).EndInit();
            ((System.ComponentModel.ISupportInitialize)(this.spnlServerName)).EndInit();
            ((System.ComponentModel.ISupportInitialize)(this.spnlVolumeName)).EndInit();
            ((System.ComponentModel.ISupportInitialize)(this.spnlUCI)).EndInit();
            ((System.ComponentModel.ISupportInitialize)(this.spnlDeviceName)).EndInit();
            ((System.ComponentModel.ISupportInitialize)(this.spnPortNumber)).EndInit();
            this.ResumeLayout(false);
            this.PerformLayout();

		}
		#endregion

		/// <summary>
		/// Set Erp Validation Error Message
		/// </summary>
		/// <param name="controlToInvalidate"></param>
		/// <param name="message"></param>
		protected void SetErpValidationErrorMessage( Control controlToInvalidate, string message )
		{
			if( !_errorProviderDirty )
				_errorProviderDirty = true;

			errError.SetVbecsError( controlToInvalidate, message ); // **** 508 Compliance ***
		}

		/// <summary>
		/// Clear Control Validation Error Message
		/// </summary>
		/// <param name="controlToClearMessageFor"></param>
		protected void ClearControlValidationErrorMessage( Control controlToClearMessageFor )
		{
			errError.SetVbecsError( controlToClearMessageFor, null ); // **** 508 Compliance ***
		}

		/// <summary>
		/// Reset Erp Validation Messages
		/// </summary>
		protected void ResetErpValidationMessages()
		{
			if( !_errorProviderDirty )
				return;

			ClearFormControlsValidationErrorMessages();

			_errorProviderDirty = false;
		}

		/// <summary>
		/// Clear Form Controls Validation Error Messages
		/// </summary>
		protected virtual void ClearFormControlsValidationErrorMessages()
		{
			// Do nothing. Child forms must override.
		}

		/// <summary>
		/// Event handler closing the form when 'Ok' button is clicked and all form data is valid.
		/// </summary>
		/// <param name="sender">Event sender object.</param>
		/// <param name="e">Standard event arguments.</param>
		private void btnOk_Click(object sender, System.EventArgs e)
		{
			if( !IsValidErrorProviderBoundCheck() )
				return;

			this.DialogResult = DialogResult.OK;
			this.Close();
		}

		/// <summary>
		/// Event handler closing the form on when 'Cancel' button is clicked.
		/// </summary>
		/// <param name="sender">Event sender object.</param>
		/// <param name="e">Standard event arguments.</param>
		private void btnCancel_Click(object sender, System.EventArgs e)
		{
			Close();
		}

		/// <summary>
		/// Helper method returning logical sum of all supplied parameters. 
		/// Differs from conventional a &amp; by forcing evaluation of all parameters
		/// even if preceding are false. 
		/// 
		/// Should be used to summarize validation results on a form.
		/// </summary>
		/// <param name="valuesToCheck">List of parameters to be evaluated and logically added.</param>
		/// <returns>Logic sum of all supplied parameters.</returns>
		protected static bool ForcedCheckLogicAnd( params bool[] valuesToCheck )
		{
			bool _result = true;

			foreach( bool _vl in valuesToCheck )
				_result = _result && _vl;

			return _result;
		}

		/// <summary>
		/// Method that is expected to be overriden by derived classes if needed. 
		/// Indicates whether all data on a form passes validation.
		/// </summary>
		/// <returns>True if all data on a form is valid. Otherwise - false.</returns>
		protected virtual bool IsValidErrorProviderBoundCheck()
		{
			return true;
		}

		/// <summary>
		/// Initializes status bar.
		/// </summary>
		private void InitStatusBar()
		{
			if( _serverInfo == null )
				return;

			AssignStatusPanelValue( this.spnlServerName, "Server", _serverInfo.ServerName );
			AssignStatusPanelValue( this.spnlVolumeName, "Volume", _serverInfo.Volume );
			AssignStatusPanelValue( this.spnlUCI, "UCI", _serverInfo.Uci );
			AssignStatusPanelValue( this.spnlDeviceName, "Device", _serverInfo.Device );
			AssignStatusPanelValue( this.spnPortNumber, "Port", _serverConnectionInfo.PortNumber.ToString() );			
		}

		/// <summary>
		/// Formats name-value pair as string and assigns it to a given status bar panel.
		/// </summary>
		/// <param name="panel"><see cref="StatusBarPanel"/> to display result on.</param>
		/// <param name="panelValueName">String name.</param>
		/// <param name="valueToAssign">String value.</param>
        private void AssignStatusPanelValue(StatusBarPanel panel, string panelValueName, string valueToAssign)
		{
		    Invoke(new MethodInvoker(delegate { panel.Text = String.Format("{0}:  {1}", panelValueName, valueToAssign); }));
		}

	    /// <summary>
		/// Event handler method handling form's first time activation.
		/// Initializes form UI and brings default control to focus.
		/// </summary>
		/// <param name="sender">Event sender object.</param>
		/// <param name="e">Standard event arguments.</param>
		private void VistASignOnDialog_Activated( object sender, System.EventArgs e )
		{
			if( !_firstTimeActivationFlag )
				return;

			_firstTimeActivationFlag = false;
			
			InitStatusBar();

			if( _initFocusPoint != null )
				_initFocusPoint.Focus();
		}

		/// <summary>
		/// Validates <see cref="TextBox"/> accepting standard VistA security code performing 
		/// validation and setting error message to error provider.
		/// </summary>
		/// <param name="txtSecurityCode"><see cref="TextBox"/> to validate.</param>
		/// <returns>True if a given <see cref="TextBox"/> contains string that can be a valid VistA security code. Otherwise - false.</returns>
		protected bool ValidateSecurityCodeTextBox( TextBox txtSecurityCode )
		{
			string _validationMessage;
			bool _isValid;
			
			_isValid = VistASecurityCode.IsValidCodeString( txtSecurityCode.Text, out _validationMessage );
			
			if( _isValid )
				ClearControlValidationErrorMessage( txtSecurityCode );
			else
				SetErpValidationErrorMessage( txtSecurityCode, _validationMessage );

			return _isValid;
		}

		private void DlgVistABase_KeyDown(object sender, System.Windows.Forms.KeyEventArgs e)
		{
			//508 We are using F6 to speak the status bar
			if (Common.LogonUser.IsVisuallyImpairedUser && e != null && e.KeyCode == Keys.F6 )
			{
				Common.Utility.Speak("Server: ");
				Common.Utility.SpeakAbbreviations(_serverInfo.ServerName);
				Common.Utility.Speak("Volume: ");
				Common.Utility.SpeakAbbreviations(_serverInfo.Volume);
				Common.Utility.Speak("UCI: ");
				Common.Utility.SpeakAbbreviations(_serverInfo.Uci);
				Common.Utility.Speak("Device: ");
				Common.Utility.SpeakAbbreviations(_serverInfo.Device);
				Common.Utility.Speak("Port: ");
				Common.Utility.SpeakAbbreviations(_serverConnectionInfo.PortNumber.ToString() );			
			}
		}		

		/// <summary>
		/// Remote server info.
		/// </summary>
		protected RemoteServerSetupAndIntroInfo ServerInfo
		{
			get
			{
				return _serverInfo;
			}
		}
	}
}
